/*
 * Year 2000 bug free utility for Commodore PC10/20-III.
 * Sets the system clock by reading the date and time
 * from the RTC. It can be also used to take the current
 * date and time from the system clock and write it to the RTC
 */

#include <stdio.h>
#include <dos.h>

#define BASE                0x2C0       /* Commodore PC10/20-III */
#define MSM6242_CD_HOLD		(1 << 0)
#define MSM6242_CD_BUSY		(1 << 1)
#define MSM6242_SECOND1     	BASE + 0x0 /* 1-second digit register */
#define MSM6242_SECOND10    	BASE + 0x1 /* 10-second digit register */
#define MSM6242_MINUTE1     	BASE + 0x2 /* 1-minute digit register */
#define MSM6242_MINUTE10    	BASE + 0x3 /* 10-minute digit register */
#define MSM6242_HOUR1       	BASE + 0x4 /* 1-hour digit register */
#define MSM6242_HOUR10      	BASE + 0x5 /* PM/AM, 10-hour digit register */
#define MSM6242_DAY1        	BASE + 0x6 /* 1-day digit register */
#define MSM6242_DAY10       	BASE + 0x7 /* 10-day digit register */
#define MSM6242_MONTH1      	BASE + 0x8 /* 1-month digit register */
#define MSM6242_MONTH10     	BASE + 0x9 /* 10-month digit register */
#define MSM6242_YEAR1       	BASE + 0xa /* 1-year digit register */
#define MSM6242_YEAR10      	BASE + 0xb /* 10-year digit register */
#define MSM6242_WEEK        	BASE + 0xc /* Week register */
#define MSM6242_CD          	BASE + 0xd /* Control Register D */
#define MSM6242_CE          	BASE + 0xe /* Control Register E */
#define MSM6242_CF          	BASE + 0xf /* Control Register F */

struct time time_str;
struct date date_str;

void msm6242_lock() {
    int cnt = 5;
    unsigned char val;
    val = inportb(MSM6242_CD);
    outportb(MSM6242_CD, val | MSM6242_CD_HOLD);

    while ((inportb(MSM6242_CD) & MSM6242_CD_BUSY) && cnt) {
        /*outportb(MSM6242_CD, val | MSM6242_CD_HOLD);*/
        delay(1);
        cnt--;
    }

    if (!cnt) {
        printf("msm6242: timed out waiting for RTC (0x%x)\n", MSM6242_CD);
    }
}

void msm6242_unlock() {
    char val;
    val = inportb(MSM6242_CD);
    outportb(MSM6242_CD, val & ~MSM6242_CD_HOLD);
}

void read_rtc_time(struct time *time_str) {
    /*lock*/
    msm6242_lock();

    /*read*/
    time_str->ti_sec = (0x0F & inportb(MSM6242_SECOND1)) + 10 * (0x0F & inportb(MSM6242_SECOND10));
    time_str->ti_min = (0x0F & inportb(MSM6242_MINUTE1)) + 10 * (0x0F & inportb(MSM6242_MINUTE10));
    time_str->ti_hour = (0x0F & inportb(MSM6242_HOUR1)) + 10 * (0x0F & inportb(MSM6242_HOUR10));

    /*unlock*/
    msm6242_unlock();
}

void read_rtc_date(struct date *date_str) {
    /*lock*/
    msm6242_lock();
    /*read*/
    date_str->da_day = (0x0F & inportb(MSM6242_DAY1)) + 10 * (0x0F & inportb(MSM6242_DAY10));
    date_str->da_mon = (0x0F & inportb(MSM6242_MONTH1)) + 10 * (0x0F & inportb(MSM6242_MONTH10));
    date_str->da_year = (0x0F & inportb(MSM6242_YEAR1)) + 10 * (0x0F & inportb(MSM6242_YEAR10));

    /*unlock*/
    msm6242_unlock();
    
    if (date_str->da_year >= 80) {
        date_str->da_year += 1900;
    } else {
        date_str->da_year += 2000;
    }
}

void set_rtc_time(struct time *time_str) {
    /*lock*/
    msm6242_lock();

    /*write*/
    outportb(MSM6242_SECOND1, time_str->ti_sec % 10);
    outportb(MSM6242_SECOND10, time_str->ti_sec / 10);
    outportb(MSM6242_MINUTE1, time_str->ti_min % 10);
    outportb(MSM6242_MINUTE10, time_str->ti_min / 10);
    outportb(MSM6242_HOUR1, time_str->ti_hour % 10);
    outportb(MSM6242_HOUR10, time_str->ti_hour / 10);

    /*unlock*/
    msm6242_unlock();
}

void set_rtc_date(struct date *date_str) {
    int year = date_str->da_year;
    if (year > 2000) {
        year -= 2000;
    } else {
        year -= 1900;
    }
    year /= 10;
    
    /*lock*/
    msm6242_lock();

    /*read*/
    outportb(MSM6242_DAY1, date_str->da_day % 10);
    outportb(MSM6242_DAY10, date_str->da_day / 10);
    outportb(MSM6242_MONTH1, date_str->da_mon % 10);
    outportb(MSM6242_MONTH10, date_str->da_mon / 10);
    outportb(MSM6242_YEAR1, date_str->da_year % 10);
    outportb(MSM6242_YEAR10, year);

    /*unlock*/
    msm6242_unlock();
}

void usage() {
    printf("Usage: msms6242 [-prw]\n");
    printf("-p: print both RTC and DOS time/date\n");
    printf("-r: read RTC and set DOS time/date\n");
    printf("-w: write DOS time/date into RTC\n");
    printf("otherwise: show usage\n");
}

main(int argc, char *argv[]) {
    if (argc < 2 || argc > 2) {
        usage();
    } else {
        if (strcmp("-p", argv[1]) == 0) {
            getdate(&date_str);
            printf("DOS date %02d-%02d-%04d\n", date_str.da_mon, date_str.da_day, date_str.da_year);
            read_rtc_date(&date_str);
            printf("RTC date %02d-%02d-%04d\n", date_str.da_mon, date_str.da_day, date_str.da_year);
            gettime(&time_str);
            printf("DOS time %02d:%02d:%02d\n", time_str.ti_hour, time_str.ti_min, time_str.ti_sec);
            read_rtc_time(&time_str);
            printf("RTC time %02d:%02d:%02d\n", time_str.ti_hour, time_str.ti_min, time_str.ti_sec);
        } else if (strcmp("-w", argv[1]) == 0) {
            /* write DOS time/date to RTC */
            gettime(&time_str);
            set_rtc_time(&time_str);
            getdate(&date_str);
            set_rtc_date(&date_str);
        } else if (strcmp("-r", argv[1]) == 0) {
            /* read RTC and set DOS time/date */
            read_rtc_time(&time_str);
            settime(&time_str);
            read_rtc_date(&date_str);
            setdate(&date_str);
        } else {
            usage();
        }
    }
}
